/*
 * File: ShimmeringShapes.java
 * ================================================================
 * An interactive piece of artwork based on randomly-generated
 * GObjects of random shapes and colors.
 */
import acm.program.*;
import acm.graphics.*;
import acm.util.*;
import java.awt.event.*;

public class ShimmeringShapes extends GraphicsProgram {
	/* The number of rows and columns in the system. */
	private static final int NUM_ROWS = 10;
	private static final int NUM_COLS = 10;
	
	/* Our preferred size. */
	public static final int APPLICATION_WIDTH = 500;
	public static final int APPLICATION_HEIGHT = 500;
	
	private static final double PAUSE_TIME = 100;
	
	/* Creates a world of random shapes. */
	public void run() {
		addRandomShapes();
		addMouseListeners();
		
		/* Randomly flicker the color of objects around the screen. */
		RandomGenerator rgen = RandomGenerator.getInstance();
		while (true) {
			/* Pick a random location and flicker the color if we hit something there. */
			double x = rgen.nextDouble(0, getWidth());
			double y = rgen.nextDouble(0, getHeight());
			GObject hit = getElementAt(x, y);
			if (hit != null) {
				hit.setColor(rgen.nextColor());
			}
			
			pause(PAUSE_TIME);
		}
	}
	
	/**
	 * Fills in the world with a random assortment of GOvals and
	 * GRects.
	 */
	private void addRandomShapes() {
		/* The width and height of each shape. */
		double width  = (double)getWidth() / NUM_ROWS;
		double height = (double)getHeight() / NUM_COLS;
		
		/* Draw a 2D grid of random shapes. */
		for (int x = 0; x < NUM_ROWS; x++) {
			for (int y = 0; y < NUM_COLS; y++) {
				GObject toAdd = createRandomObject(x * width, y * height, width, height);
				add(toAdd);
			}
		}
	}
	
	/**
	 * Creates a random shape at the given location having
	 * the given width and height.
	 * @param x The x location at which to draw the shape.
	 * @param y The y location at which to draw the shape.
	 * @param width The width of the shape.
	 * @param height The height of the shape.
	 * @return The created object.
	 */
	private GObject createRandomObject(double x, double y, double width, double height) {
		RandomGenerator rgen = RandomGenerator.getInstance();
		
		/* Randomly decide whether to make a GOval or a GRect. */
		if (rgen.nextBoolean()) {
			return createOval(x, y, width, height);
		} else {
			return createRect(x, y, width, height);
		}
	}

	/**
	 * Creates a filled, black GOval with the given size and location
	 * information.
	 * @param x The oval's x coordinate.
	 * @param y The oval's y coordinate.
	 * @param width The oval's width.
	 * @param height The oval's height.
	 * @return The created oval.
	 */
	private GOval createOval(double x, double y, double width, double height) {
		GOval oval = new GOval(x, y, width, height);
		oval.setFilled(true);
		return oval;
	}
	
	/**
	 * Creates a filled, black GRect with the given size and location
	 * information.
	 * @param x The rectangle's x coordinate.
	 * @param y The rectangle's y coordinate.
	 * @param width The rectangle's width.
	 * @param height The rectangle's height.
	 * @return The created rectangle.
	 */
	private GRect createRect(double x, double y, double width, double height) {
		GRect rect = new GRect(x, y, width, height);
		rect.setFilled(true);
		return rect;
	}
	
	/**
	 * Changes the color of the object under the mouse when the mouse moves,
	 * assuming there is an object there.
	 */
	public void mouseMoved(MouseEvent e) {
		/* Find what object is below the cursor. */
		GObject hovered = getElementAt(e.getX(), e.getY());
		
		/* If something is there, randomly change the color. */
		if (hovered != null) {
			RandomGenerator rgen = RandomGenerator.getInstance();
			hovered.setColor(rgen.nextColor());
		}
	}
	
	/**
	 * Toggles the type of object, if any, that's below the cursor.
	 */
	public void mouseClicked(MouseEvent e) {
		GObject hovered = getElementAt(e.getX(), e.getY());
		if (hovered != null) {
			/* Get rid of the old object. */
			remove(hovered);
			
			/* Using the instanceof keyword, determine what we hit and replace it with an object
			 * of the opposite type.
			 */
			if (hovered instanceof GRect) {
				add(createOval(hovered.getX(), hovered.getY(), hovered.getWidth(), hovered.getHeight()));
			} else if (hovered instanceof GOval) {
				add(createRect(hovered.getX(), hovered.getY(), hovered.getWidth(), hovered.getHeight()));
			}
		}
	}
}
